/*
   (C) 2023 R.Schick / Thorsten Gurzan - beelogger.de

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

// beelogger.de - Arduino Datenlogger für Imker
//

//----------------------------------------------------------------
// Allgemeine Konfiguration
//----------------------------------------------------------------

// #define UNI_2X  // Universal Platine Version 2.x // am Zeilenanfang entfernen


// Name des WLAN-Zugangspunkts
const char  Access_Point[] PROGMEM = "WLAN_Name";

// Passwort des Zugangspunktes
const char AP_Passwort[] PROGMEM =   "WLAN_Passwort#";

// Domainname zum Webserver mit beelogger-Skript, Bsp:"meineDomain.de";
const char serverName[] PROGMEM = "community.beelogger.de";

// GET mit Verzeichnis  Webserver-Skript und das PHP Skript für den jeweiligen beelogger
// Bsp: "GET /USER_X/beelogger6/beelogger_log.php?"

const char beelogger_pfad[] PROGMEM = "GET /USER/beelogger1/beelogger_log.php?";   //USERX = meinOrdner bzw. Name, Y = beelogger Nummer  1...n

// Passwort vom Webserver-Skript
const char Passwort[] PROGMEM =  "Log"; // Log = Default

//----------------------------------------------------------------
// Ende Konfiguration
//----------------------------------------------------------------

// EE-Prom Adresse
uint8_t _EEProm_adress = 0x57;
// DS3231 Adresse
#define _DS3231_ADDRESS 0x68

char ID_ID[] = "WLAN_TEST_240503";

//----------------------------------------------------------------
// Pin-Belegung beelogger
//----------------------------------------------------------------
#define Modul_CS      10    // Ethernet abschalten
#define Power_Pin      4    // Power On

#define Batterie_messen    A6
#define Solarzelle_messen  A7

#if (defined(ARDUINO_AVR_PRO) && (F_CPU==16000000L) && (not defined UNI_2X)) // Pro Mini 5V 16MHz Universal 1.x  
#undef Batterie_messen
#undef Solarzelle_messen
#define Batterie_messen    A2
#define Solarzelle_messen  A3
#endif
//----------------------------------------------------------------

#define Seriell_Baudrate 9600

//----------------------------------------------------------------
// ESP-8266 WLAN Modul
//----------------------------------------------------------------
#define ESP_Baudrate 9600
uint8_t ESP_RX = 8;
uint8_t ESP_TX = 9;

#define ESP_RESET A2

#if (defined(ARDUINO_AVR_PRO) && (F_CPU==16000000L) && (not defined UNI_2X)) // Pro Mini 5V 16MHz Universal 1.x 
#undef ESP_RESET
#define ESP_RESET 12
#endif

#include "ESP_beelogger.h"
C_WLAN_ESP01 dfue;

const char Str_Http[]  PROGMEM = " HTTP/1.1\r\n";
const char Str_Con[]   PROGMEM = "Connection: close\r\n\r\n";
#define SERVER_TM_OUT 10000 // 10000 = 10 sec
const char Str_Pw[]      PROGMEM = {"PW="};
const char Str_Opt[]     PROGMEM = {"&A=1"};
const char Str_Id[]      PROGMEM = {"&ID="};
const char Str_MData[]   PROGMEM = {"&M1_Data="};  // Parameter Multi Data  Senden
//----------------------------------------------------------------


/****************************************************************/
// Variablen
/****************************************************************/
const float No_Val = 99.9f;  // Vorbelegung, Wert nicht gemessen
const float No_Value = -1.0f;  // Vorbelegung, Wert nicht gemessen

// Test-Messdaten (regulär kommen die Messdaten von den Sensoren)
float SensorTemp[2] = {25.0, 35.0};
float SensorFeuchte[2] = {40.0, 60.0};
float Licht = 5000;
float Gewicht[2] = {2.5, No_Value};
float Batteriespannung = 4.7;
float Solarspannung = 1.7;
float Service = 0.0;
float Aux[3] = {1.0, 2.0, 3.0};
#define Aussenwerte 0
#define Beute1      1


uint8_t OK = true;
char DatenString[96];  // Datenbuffer
char parameter[96], data[64];
uint8_t first_run = 1;
uint8_t Ds3231_ok = 0;

#include <Wire.h>
#include "Sodaq_DS3231.h"

void setup() {
  digitalWrite (10, HIGH); //HX711 aus
  pinMode(10, OUTPUT);
  digitalWrite (A0, HIGH); //HX711 aus
  pinMode(A0, OUTPUT);

  digitalWrite(Modul_CS, HIGH);   // Modul Ethernet usw. aus
  pinMode(Modul_CS, OUTPUT);

  digitalWrite(Power_Pin, HIGH);  // Power an
  pinMode(Power_Pin, OUTPUT);

  digitalWrite(ESP_RESET, HIGH);  // ESP8266 an
  pinMode(ESP_RESET, OUTPUT);

  Serial.begin(Seriell_Baudrate);
  while (!Serial ) {};
  delay(100);
  Serial.print(F("ESP 8266 Test Sketch "));
  Serial.println(ID_ID);
  Serial.flush();
  delay(1000);


  Serial.println(F("\nTeste DS3231 vorhanden?"));
  Serial.flush();
  Wire.begin();
  Wire.beginTransmission(_DS3231_ADDRESS);
  uint8_t error = Wire.endTransmission();

  if (error == 0) {
    Ds3231_ok = 1;
    Serial.println(F("DS3231 ... gefunden\n"));
    EE_WriteMem(0, ID_ID, strlen(ID_ID));
    EE_ReadStr(0, DatenString, strlen(ID_ID));
    //Serial.println(DatenString);
    if (strcmp(ID_ID, DatenString) == 0)Serial.println(F("EE-Prom ... i.O.\n"));
    else {
      Serial.println(F("###  EE-Prom Fehler  ###\n"));
      delay(10000);
    }
    rtc.begin();
  }
  else {
    Serial.println(F("nein\n"));
  }
  Serial.print(F("Abfrage ESP8266 ... "));
  if (dfue.init(ESP_Baudrate)) {
    Serial.println(F("i.O."));
    //    dfue.mode();      // Station Mode permanent
    Serial.println(" ");
    Serial.println(F("ESP Firmware:"));
    dfue.firmware();  // firmware abfragen
    Serial.println(dfue.buffer);
    Serial.println(" ");
    Serial.flush();
    WLan_Info();
    //WLan_Pegel();
  }
  else {
    OK = false;
    Serial.println(F("\nModul konnte nicht abgefragt werden"));
	Serial.println(F("  Konfiguration 9600Bd ?"));
    Serial.println(dfue.buffer);
    Serial.println(" ");
    Serial.flush();
    ESP_Abschaltung();
  }

  strcpy_P(parameter, beelogger_pfad);

  char *p_buf = strstr(parameter, "/beelogger");  // search first /
  uint8_t n = atoi(p_buf + 10);
  //Serial.println(p_buf + 10);
  if (n == 0) {
    Serial.println(parameter);
    Serial.println(F("\nWLan Test Sketch nur mit beeloggerX,  X = 1...n"));
    Serial.println(F("keine DUO, Triple, Quad, usw."));
    Serial.flush();
    OK = false;
    ESP_Abschaltung();
  }

  if (OK) {
    strcpy_P(parameter, Access_Point);
    Serial.print(F("Verbinde zum WLAN: "));
    Serial.print(parameter);
    Serial.println(F(" ... "));
    Serial.flush();
    strcpy_P(data, AP_Passwort);

    if (dfue.join(parameter, data)) {
      Serial.println(F("WLAN Verbindung OK"));
      Serial.println("\n");
      Serial.flush();
    }
    else {
      OK = false;
      Serial.println(F("Verbindungsfehler!\n"));
      Serial.flush();
      ESP_Abschaltung();
    }
  }

  if (OK) {
    Serial.println(F("Verbinde mit Server ... "));
    strcpy_P(parameter, serverName);
    Serial.print(parameter);
    if (dfue.Connect(parameter)) {
      Serial.println(F("  ... connect: OK\n"));
      Serial.flush();
    }
    else {
      OK = false;
      Serial.println(F(" connect fehlgeschlagen\n"));
      Serial.println(dfue.buffer);
      Serial.println(" ");
      Serial.flush();
      ESP_Abschaltung();
    }
  }
}



uint8_t count = 0;
//#########################################
void loop() {

  if ((!first_run) && (OK)) {
    Serial.print(F("ESP ... "));
    Serial.flush();
    if (dfue.init(ESP_Baudrate)) {  // baud rate
      Serial.println(F("o.k."));
      Serial.flush();
      strcpy_P(parameter, Access_Point);
      strcpy_P(data, AP_Passwort);
      if (dfue.join(parameter, data)) {
        Serial.println(F(".."));
        Serial.flush();
        strcpy_P(parameter, serverName);
        if (dfue.Connect(parameter)) {
          Serial.println(".");
          Serial.flush();
          OK = true;
        }
      }
    }
  }
  else  first_run = 0;

  Batteriespannung = Messe_Spannung(Batterie_messen);

  if (OK) {

    if (dfue.prep_send(300)) { // fiktive Länge, senden wird mit 0x00 gestartet
      Serial.println(F("Uebertrage Daten zum Webserver ..."));
      Serial.flush();
      DatenString_erstellen();
      //Serial.println(DatenString);
      //Serial.flush();
      //"GET " + url + php + "?" + Daten + " HTTP/1.1\r\n" + "Host: " + host + "\r\n" + "Connection: close\r\n\r\n");
      // Wichtig: der String darf von der url bis Ende der Daten keine Leerzeichen enthalten

      strcpy_P(parameter, beelogger_pfad);    // GET ....
      dfue.send(parameter);
      Serial.println(parameter);
      Serial.flush();
      strcpy_P(parameter, Str_Pw);
      dfue.send(parameter);
      strcpy_P(parameter, Passwort);
      dfue.send(parameter);
      strcpy_P(parameter, Str_Opt);         //
      dfue.send(parameter);
      dfue.send("&Z=2");

      // System ID
      strcpy_P(parameter, Str_Id);
      dfue.send(parameter);
      strcpy(parameter, ID_ID);
      dfue.send(parameter);
      delay(50);

      //Daten
      Serial.println(DatenString);
      Serial.flush();
      strcpy_P(parameter, Str_MData);
      dfue.send(parameter);
      delay(50);

      dfue.send(DatenString);  // Datensatz
      delay(50);

      strcpy_P(parameter, Str_Http);
      dfue.send(parameter);
      dfue.send("Host: ");
      strcpy_P(parameter, serverName);
      dfue.send(parameter);
      dfue.send("\r\n");

      strcpy_P(parameter, Str_Con);
      dfue.send(parameter);
      dfue.send(0x00); // Startkommando senden

      OK = false;
      if (dfue.sendCommand(0, 2000, "200 OK")) {
        Serial.println(F("HTTP OK"));
        if (dfue.sendCommand(0, 5000, "ok *")) {
          OK = true;

          Serial.print(dfue.buffer);
          Serial.println(" ");
          Serial.flush();

          char *p_buf = strstr(dfue.buffer, "ok *");  // search string start
          uint8_t pos = 0;
          *p_buf = 0; //Stringende
          do {
            p_buf --;    // search start of line
            pos++;
          }
          while ( (!(*p_buf == '\n')) && (pos < 20));

          int n;
          if (char *p_bf = strchr(p_buf, 'T') ) {
            n = atoi(p_bf + 1);  // Konvertiere str in int

            char x = *(p_buf + 1);
            if ((x > 0x2F) && (x < 0x3A)) { // first char a number?
              long l_tm = atol(p_buf);
              l_tm = l_tm - 946684800;  // EPOCH_TIME_OFFSET
              if (Ds3231_ok == 1) {
                rtc.setDateTime(l_tm);
                Serial.println(F("\nUhrzeit in DS3231 gesetzt.  "));
                display_time();
              }
            }
          }
          else {
            n = atoi(p_buf); // Konvertiere str in int
          }
          if (n > 0) {
            Serial.print(F("\nIntervallvorgabe vom Server:  "));
            Serial.print(n);
            Serial.println(F(" Minuten "));
          }
          else {
            Serial.print(F("Kein Intervall vom Server vorgegeben. "));
            Serial.println(dfue.buffer);
            Serial.println(" ");
            Serial.println(n);
          }
          Serial.print(F("Vorgabe Sendezyklus vom Server:  "));
          uint8_t std = 0;
          if (strchr(p_buf, 'A') ) {
            std = 1; Serial.print(" eine ");
          }
          else if (strchr(p_buf, 'B') ) {
            std = 1; Serial.print(" zwei ");
          }
          else if (strchr(p_buf, 'C') ) {
            std = 1; Serial.print(" vier ");
          }
          else if (strchr(p_buf, 'D') ) {
            std = 1; Serial.print(" acht ");
          }
          if (std) {
            Serial.println(F(" Stunde(n)"));
          } else {
            Serial.print(F(" nein "));
          }
          Serial.println(" ");
          Serial.println(F("\nVerbindung erfolgreich"));
          Serial.println(" ");
        }
        else {
          Serial.println(F("Quittung Fehler!"));
          Serial.println(dfue.buffer);
          OK = false;
        }
      }
      else {
        Serial.println(F("\nFehler in:"));
        strcpy_P(parameter, serverName);
        Serial.println(parameter);
        strcpy_P(parameter, beelogger_pfad);
        Serial.println(parameter);
        Serial.flush();
      }
      if (dfue.quit()) {
        Serial.println(F("."));
      }
      else {
        OK = false;
      }
      Serial.flush();
      delay(2000);
    } // prep_send

    if (OK) {
      // Test-Messdaten um 1 erhöhen
      SensorTemp[0]++;
      SensorTemp[1] += 2;
      SensorFeuchte[0]++;
      SensorFeuchte[1]++;
      Licht++;
      Gewicht[0]++;
      Service = 1.0;

      // Abschaltung nach 5 Datenübertragungen
      count++;
      if (count >= 2) {
        OK = false;
        ESP_Abschaltung();
      }
      Serial.println(F(" ... Senden in 5 Minuten"));
      Serial.flush();
      for (int i = 0; i < 300; i++) {
        delay(1000);
      }
    }
    else {
      Serial.println(F("Fehler im Test!"));
      ESP_Abschaltung();
    }
  } // if OK
}
//#########################################

#include <avr/sleep.h>
void ESP_Abschaltung() {
  Serial.println(F("Test beendet."));
  Serial.println(" ");
  dfue.disConnect();
  dfue.quit();
  delay(100);
  dfue.end();
  Serial.end();
  digitalWrite (A0, LOW); //HX711 aus
  digitalWrite (10, LOW); //HX711 aus
  digitalWrite(Power_Pin, LOW);  // Power aus
  digitalWrite(ESP_RESET, LOW);  // ESP8266 aus
  set_sleep_mode(SLEEP_MODE_PWR_SAVE);
  cli();
  sleep_enable();
  sei();
  sleep_cpu();
  sleep_disable();
}
//----------------------------------------
void WLan_Info() {
  uint8_t st;
  Serial.println(F("WLAN Information:\n"));
  Serial.flush();
  st = dfue.espState;
  dfue.espState = ESP_CONNECTED;
  dfue.send("AT+CWLAP\r\n");
  dfue.check(10000);
  dfue.espState = st;
  Serial.println("\n");
  Serial.flush();

  delay(1000);
}
//----------------------------------------
void WLan_Pegel() {
  strcpy_P(parameter, Access_Point);
  Serial.print(parameter);
  Serial.print(F("   Signalstaerke: "));
  Serial.flush();
  if (dfue.signal(parameter)) {         // hole Signalstärke
    strcpy(parameter, dfue.buffer);
    char cc = ',';
    char *pos = strchr(parameter, cc); // suche ','
    if (pos) {
      int svc = atoi(pos + 1);
      if ((svc > -10)) {
        Serial.println(F("\n###########################"));
        Serial.println(F("Signalstärke unklar: "));
      }
      if ((svc < -88)) {
        Serial.println(F("\n###########################"));
        Serial.println(F("Signal schwach: "));
      }
      Service = (float)svc;
      Serial.print(svc);
      Serial.println(F(" dBm\n"));
      Serial.flush();
    }
  }
  else {
    Serial.println(F("\n###########################"));
    Serial.println(F("Hinweis: Signalabfrage wird von ESP-Firmwareversion nicht unterstützt"));
  }
  Serial.flush();
}
//----------------------------------------------------------------
// Uhrzeit anzeigen
//----------------------------------------------------------------
void  display_time() {
  DateTime jetzt = rtc.now();
  if (jetzt.year() > 2100) {
    Serial.println(F("\nDS3231?"));
  }
  else {
    Serial.print(jetzt.date());
    Serial.print('.');
    Serial.print(jetzt.month());
    Serial.print('.');
    Serial.print(jetzt.year());
    uint8_t h = jetzt.hour();
    if (h < 10) {
      Serial.print("  ");
    }
    else Serial.print(" ");
    Serial.print(h);
    Serial.print(':');
    if (jetzt.minute() < 10) Serial.print("0");
    Serial.print(jetzt.minute());
    Serial.print(':');
    if (jetzt.second() < 10) Serial.print("0");
    Serial.print(jetzt.second());
  }
  Serial.println(" ");
  Serial.flush();
}

/****************************************************************/
// Pins Spannungen messen
/****************************************************************/
/****************************************************************/
// Teilerwerte für die Spannungsmessung
/****************************************************************/
const long Kalib_Spannung = 4000;
const long Kalib_Bitwert  = 1500;
/****************************************************************/


float Messe_Spannung (uint8_t Pin) {
  int Messung_Spannung;
  float Spannung;
  Messung_Spannung = analogRead(Pin);
  Messung_Spannung = 0;
  for (uint8_t j = 0 ; j < 16; j++) {
    Messung_Spannung += analogRead(Pin);
  }
  Messung_Spannung = Messung_Spannung >> 2;
  Spannung = (float)map(Messung_Spannung, 0, Kalib_Bitwert, 0, Kalib_Spannung) / 1000.0;
  return (Spannung);
}
/**
  @brief Funktion DatenString_erstellen()
  erstellt konfigurationsabhängigen Datensatz zum abspeichern im EE-Prom

  @param  none,        globale Variablen
  @return  none
*/
void DatenString_erstellen() {

  int count = 0;
  DateTime jetzt = DateTime(__DATE__, __TIME__);
  if (Ds3231_ok == 1) jetzt = rtc.now();
  count = sprintf(DatenString, "%d/", jetzt.year());
  count += sprintf(DatenString + count, "%2.2d/", jetzt.month());
  count += sprintf(DatenString + count, "%2.2d_", jetzt.date());
  count += sprintf(DatenString + count, "%2.2d:", jetzt.hour());
  count += sprintf(DatenString + count, "%2.2d:", jetzt.minute());
  count += sprintf(DatenString + count, "%2.2d",  jetzt.second());

  count = Wert_hinzufuegen(count, SensorTemp[Beute1], 1, No_Val);    // Stocktemperatur 1
  count = Wert_hinzufuegen(count, SensorTemp[Aussenwerte], 1, No_Val);     // Außentemperatur
  count = Wert_hinzufuegen(count, SensorFeuchte[Beute1], 1, No_Value);     // Stockluftfeuchte 1
  count = Wert_hinzufuegen(count, SensorFeuchte[Aussenwerte], 1, No_Value);  // Außenluftfeuchte
  count = Wert_hinzufuegen(count, Licht, 1, No_Value);                 // Licht
  count = Wert_hinzufuegen(count, Gewicht[0], 2, No_Value);            // Gewicht 1

  count = Wert_hinzufuegen(count, Batteriespannung, 2, No_Value); // Akkuspannung
  count = Wert_hinzufuegen(count, Solarspannung, 2, No_Value);    // Solarspannung
  count = Wert_hinzufuegen(count, Service, 2, No_Value);    // Service
  count = Wert_hinzufuegen(count, Aux[1], 1, No_Value);  // Aux 1
  count = Wert_hinzufuegen(count, Aux[2], 2, No_Value);  // Aux 2
  count = Wert_hinzufuegen(count, Aux[0], 2, No_Value);  // Aux 0
  DatenString[count] = 0;
}


/**
  @brief Funktion Wert_hinzufuegen()
  erstellt Stringanteil für Datensatz

  @param
  count: aktuelle Position im String,
  Wert: Messwert,
  Nachkommastellen: Anzahl der Nachkommastellen für diesen Messwert
  Fehler: Vergleichswert für fehlerhaften oder unbenutzten Messwert
  @return  neue Position im Datenstring
*/
int Wert_hinzufuegen(int count, float Wert, uint8_t Nachkommastellen, float Fehler) {
  char Konvertierung[16];
  int count_neu = count;

  if (Wert == Fehler) {
    count_neu += sprintf(DatenString + count, ",%s", "");
  } else {
    dtostrf(Wert, 1, Nachkommastellen, Konvertierung);
    count_neu += sprintf(DatenString + count, ",%s", Konvertierung);
  }
  return count_neu;
}
//--------------------------------------------------

// EE-Prom functions
uint8_t EE_WriteMem(int iAddr, const char *pBuf, int iCnt)
{
  uint8_t iBytes, iRC = 0;
  // Writes are restricted to a single 32 byte page.  Therefore. if a write spans a page
  // boundry we must split the write.
  while (iCnt > 0) {
    iBytes = min(iCnt, BUFFER_LENGTH - 2);
    int iCurPage = iAddr & ~((int)0x1f);
    if (iAddr + iBytes > iCurPage + 32) { // Number of bytes is too large
      iBytes = (iCurPage + 32) - iAddr;
    }
    Wire.beginTransmission(_EEProm_adress);
    Wire.write( highByte(iAddr) ); // Address MSB
    Wire.write( lowByte(iAddr) );  // Address LSB
    Wire.write((uint8_t*)pBuf, iBytes);
    Wire.endTransmission();
    iRC  += (int)iBytes;
    iCnt -= (int)iBytes;
    iAddr += (int)iBytes;
    pBuf += (int)iBytes;
    delay(10); // delay(50);  // Give the EEPROM time to write its data
  }  /* while */
  delay(20);
  return (iRC);
}

uint8_t EE_ReadStr(int iAddr, char Buf[], int iCnt) {
  uint8_t iRead = 0, iBytes;
  char c;
  while (iCnt > 0) {
    Wire.beginTransmission(_EEProm_adress);
    Wire.write(iAddr >> 8); // Address MSB
    Wire.write(iAddr & 0xff); // Address LSB
    Wire.endTransmission();
    iBytes = min(iCnt, 128);
    Wire.requestFrom(_EEProm_adress, iBytes);
    while (Wire.available() && iCnt > 0) {
      c = Wire.read();
      Buf[iRead] = c;
      if (c == '\0') {
        iCnt = 0; break;
      }  /* if */
      iRead++; iCnt--; iAddr++;
    }  /* while */
  }
  return (iRead);
}
