/*
   (C) 2023 R.Schick / Thorsten Gurzan - beelogger.de

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

// beelogger.de - Arduino Datenlogger für Imker
// Erläuterungen dieses Programmcodes unter http://beelogger.de


//----------------------------------------------------------------
// Konfiguration
//----------------------------------------------------------------

// #define UNI_2X  // Universal Platine Version 2.x // am Zeilenanfang entfernen

// Kalibrierung, bitte Werte eintragen.
//----------------------------------
int Kalib_Spannung = 4000; // Hier muss der Wert aus der Kalibrierung eingetragen werden
int Kalib_Bitwert  = 1500; // Hier muss der Wert aus der Kalibrierung eingetragen werden
//----------------------------------

//----------------------------------------------------------------
// Ende Konfiguration
//----------------------------------------------------------------



#include "Sodaq_DS3231.h"
#include <Wire.h>
#include <LowPower.h>

#define SERIAL_BAUDRATE 9600

// Intervalle
//----------------------------------
uint8_t WeckIntervallMinuten = 2;
uint8_t AlternativIntervallMinuten = 6;
float VAlternativ = 3.8;           // Minimale Spannung ab der automatisch das alternative Intervall aktiviert wird
float VMinimum = 3.9;              // Minimale Spannung, Fehlermeldung
//----------------------------------


float Batteriespannung = -1.0;
float Solarspannung = -1.0;
float RTCTemp = 99.9;

volatile bool ok_sleep = false;

#define USB_RX      0          // Pin 0 RX-USB
#define USB_TX      1          // Pin 1 TX-USB
#define DS3231_Interrupt_Pin  2

#define Power_Pin  4

#define Stop_GSM_WLAN      A2
#define Batterie_messen    A6
#define Solarzelle_messen  A7

#if (defined(ARDUINO_AVR_PRO) && (F_CPU==16000000L) && not defined (UNI_2X)) // Pro Mini 5V 16MHz  Universal Platine 1.x
#undef Stop_GSM_WLAN
#define Stop_GSM_WLAN      12
#undef Batterie_messen
#define Batterie_messen    A2
#undef Solarzelle_messen
#define Solarzelle_messen  A3
#endif


DateTime aktuell;
uint8_t count = 0;
//--------------------------------------------------
void setup() {
  Serial.begin(SERIAL_BAUDRATE);
  while (!Serial) {};
  Serial.println(F("beelogger Power-On-Off-Test 08.12.2023"));
  Serial.println(" ");

#if defined (UNI_2X)
  Serial.println(F(" beelogger-Universal Platine 2.x"));
#else
#if defined(ARDUINO_AVR_PRO)// Pro Min
#if (F_CPU==16000000L)     // 5V 16MHz
  Serial.print(F(" beelogger-Universal 1.x mit Arduino Pro-Mini 5V / 16MHz "));
  Serial.println(F(" 5V / 16MHz  "));
#else
  Serial.print(F(" beelogger-SMD  mit Arduino Pro-Mini 3,3V 8MHz "));
  Serial.println(F(" 3,3V / 8MHz  "));
#endif
#endif
#if defined(ARDUINO_AVR_NANO)
  Serial.println(F(" beelogger-Universal 1.x mit Arduino Nano  "));
#endif
#endif

  Serial.println(" ");
  Serial.println(F("... teste: Uhr(DS3231)"));
  Serial.println(F("           ADC-Kalibrierwerte"));
  Serial.println(F("           und Power - ON / OFF "));
  Serial.println(" ");

  digitalWrite(Stop_GSM_WLAN, LOW);  // On hold
  pinMode(Stop_GSM_WLAN, OUTPUT);
  delay(5);

  pinMode(Power_Pin, OUTPUT);
  digitalWrite(Power_Pin, HIGH);  // System On
  delay(5);

  Serial.println("Starte Uhrbaustein:");
  Serial.flush();
  Wire.begin();
  Wire.beginTransmission(0x68); //DS3231
  uint8_t error = Wire.endTransmission();
  if (error) {
    Serial.println(F("\nUhrbaustein (DS3231) nicht gefunden!"));
    Serial.println(F("\n STOP!"));
    digitalWrite(Power_Pin, LOW);
    while (1) {
      delay(100);
    };
  }

  rtc.begin();

  DateTime pc_tim = DateTime(__DATE__, __TIME__);
  long l_pczeit = pc_tim.get();
  DateTime aktuell = rtc.now();
  long l_zeit = aktuell.get();
  if (l_pczeit > l_zeit)  rtc.setDateTime(pc_tim.get());

  rtc.clearINTStatus();

  delay(100);
  pinMode(DS3231_Interrupt_Pin, INPUT_PULLUP);
  delay(100);

  check_D2();  // check Interrupt Pin

  pinMode(DS3231_Interrupt_Pin, INPUT_PULLUP);
  ok_sleep = true;

  delay(5);

  if (display_time() == 0) {
    Serial.println(F("\n\nSketch STOP"));
    digitalWrite(Power_Pin, LOW);
    while (true) {
      delay(100);
    };
  }
  display_temp();

  if ( Kalib_Bitwert == 100) {
    Serial.println(" \n");
    Serial.println(F("Bitte ADC Kalibrierwerte eintragen! "));
    Serial.println(F("Bitte ADC Kalibrierwerte eintragen! "));
    Serial.println(F("Bitte ADC Kalibrierwerte eintragen! "));
    Serial.println(" \n");
  }
  Spannungen_messen();

  WeckIntervallMinuten = 1; //Power-On-Test
  Serial.print(F("\nTeste  Power ON:"));
  Alarm_konfigurieren();
  WeckIntervallMinuten = 2; // Power-Off-Test

  SleepNow(false);
}
//--------------------------------------------------


//##################################################
void loop() {
  check_intervall();
  check_D2();  // check Interrupt Pin
  display_time();
  Spannungen_messen();

  Serial.print(F("\n\nTeste  Power OFF:"));
  Alarm_konfigurieren();
  SleepNow(true);
}
//##################################################

void check_D2() {
  int val = digitalRead(DS3231_Interrupt_Pin);
  if ( val == 0) {
    Serial.println(" \n");
    Serial.println(F("  PIN D2  'ist auf' 0 Volt, eventuell: "));
    Serial.println(F("   - Modifikation DS3231-Modul nicht erfolgt oder fehlerhaft? "));
    Serial.println(F("   - Kurzschluß an D2?  Arbeits-Schalter an D2 ein? "));
    Serial.println(F("  Sketch nicht funktionsfähig! "));
    Serial.println(" ");
    Serial.println(F("  Sketch Neustart versuchen! "));
    Serial.println(" \n");
    while (digitalRead(DS3231_Interrupt_Pin) == 0) {
      delay (5000);
      Serial.print(".");
    };
  }
}
//--------------------------------------------------
uint8_t  display_time() {
  uint8_t ok_val = 0;
  char buf[8];
  DateTime aktuell = rtc.now();
  if ( aktuell.year() > 2100) {
    Serial.print(F("\nDS3231 Uhrbaustein nicht erkannt oder nicht vorhanden \n"));
  }
  else {
    ok_val = 1;
    Serial.print(F("beelogger-System Datum und Uhrzeit: "));
    Serial.print(aktuell.date());
    Serial.print('.');
    Serial.print(aktuell.month());
    Serial.print('.');
    Serial.print(aktuell.year());
    byte h = aktuell.hour();
    if (h < 10) {
      Serial.print("  ");
    }
    else Serial.print(" ");
    Serial.print(h);
    Serial.print(':');
    sprintf(buf, "%02d", aktuell.minute());
    Serial.print(buf);
    Serial.print(':');
    sprintf(buf, "%02d", aktuell.second());
    Serial.print(buf);
  }
  Serial.println(" ");
  Serial.flush();
  return (ok_val);
}
//--------------------------------------------------

//--------------------------------------------------
void check_intervall() {
  long l_akt = aktuell.get() - 60; //min. 1 minute
  DateTime jetzt = rtc.now();
  long l_jetzt = jetzt.get();
  long df = (l_jetzt - l_akt);
  if (df < 55) {
    Serial.println(F("********************************************"));
    Serial.println(F("Fehler Sleepzeit nicht erreicht !"));
    Serial.println(F(" Test wiederholen, wenn Fehler nochmal erfolgt:"));
    Serial.println(F("  DS3231-Modul  Modifikation nicht erfolgt oder fehlerhaft? "));
    Serial.println(F("********************************************"));
    Serial.println(" ");
    Serial.flush();
  }
}
//--------------------------------------------------

//--------------------------------------------------
void display_temp() {
  RTCTemp = rtc.getTemperature();
  Serial.print(F("Temperatur ueber Sensor in RTC: "));
  Serial.println(RTCTemp);
  Serial.println(" ");
  Serial.flush();
}
//--------------------------------------------------



//--------------------------------------------------
void Spannungen_messen() {
  Serial.print(F("\n Batterie:      "));
  Batteriespannung = Messe_Spannung(Batterie_messen);
  Serial.print(Batteriespannung);
  Serial.println(F(" V"));
  Serial.flush();
  Serial.print(F(" Solarspannung: "));
  Solarspannung = Messe_Spannung(Solarzelle_messen);
  Serial.print(Solarspannung);
  Serial.println(F(" V"));
  Serial.println(" ");
  Serial.flush();
}

float Messe_Spannung (byte Pin) {
  int Messung_Spannung;
  float Spannung;
  Messung_Spannung = analogRead(Pin);
  Messung_Spannung = 0;
  for (byte j = 0 ; j < 16; j++) {
    Messung_Spannung += analogRead(Pin);
  }
  Messung_Spannung = Messung_Spannung >> 2;
  Serial.print(F("  ADC-Bitwert= "));  Serial.print(Messung_Spannung); Serial.print("\t\t");
  Spannung = (float)map(Messung_Spannung, 0, Kalib_Bitwert, 0, Kalib_Spannung) / 1000.0;
  return (Spannung);
}
//--------------------------------------------------


//--------------------------------------------------
void Alarm_konfigurieren() {
  byte IntervallMinuten;

  if (Batteriespannung > VAlternativ) {
    IntervallMinuten = WeckIntervallMinuten;
  } else {
    Serial.println(" ");
    Serial.println(F("******************************************************************"));
    Serial.println(F("Batteriespannung unter Grenzwert, alternatives Intervall aktiviert"));
    Serial.println(F("******************************************************************"));
    Serial.println(" ");
    IntervallMinuten = AlternativIntervallMinuten;
  }
  Serial.println(" ");
  Serial.print(F("Weckintervall: "));
  Serial.print(IntervallMinuten);
  Serial.println(F(" Minute(n)"));
  Serial.flush();

  aktuell = rtc.now();
  long timestamp = aktuell.get();
  aktuell = timestamp + IntervallMinuten * 60;
  rtc.enableInterrupts(aktuell.hour(), aktuell.minute(), aktuell.second());

  Serial.print(F("Wakeup at: "));
  Serial.print(aktuell.hour());
  Serial.print(":");
  if (aktuell.minute() < 10) Serial.print("0");
  Serial.print(aktuell.minute());
  Serial.print(":");
  if (aktuell.second() < 10) Serial.print("0");
  Serial.println(aktuell.second());
  Serial.println(" ");
  Serial.flush();
}
//--------------------------------------------------


//--------------------------------------------------
void WakeUp() {
  detachInterrupt(0);
  ok_sleep = false;
}
//--------------------------------------------------


//--------------------------------------------------
void SleepNow(byte power_off) {
  delay(100);
  if (!power_off) {
    Serial.println(F("Sleep-Modus mit Power-ON gestartet, bitte warten ... "));
  }
  else {
    Serial.println(F("Sleep-Modus mit Power-OFF ist aktiviert, bitte warten ... "));
    Serial.flush();
    digitalWrite(Power_Pin, LOW);
    delay(2000);
    check_D2();  // check Interrupt Pin
  }
  Serial.flush();
  Serial.end();        // Serial aus
  delay(50);
  digitalWrite(USB_RX, LOW);   // Port aus
  pinMode(USB_TX, INPUT);
  digitalWrite(USB_TX, LOW);   // Port aus

  TWCR &= ~(bit(TWEN) | bit(TWIE) | bit(TWEA));
  digitalWrite (A4, LOW);
  digitalWrite (A5, LOW);

  delay(1);
  attachInterrupt(0, WakeUp, LOW);
  if (ok_sleep) {
    LowPower.powerDown(SLEEP_FOREVER, ADC_OFF, BOD_OFF);
  }
  else {
    Serial.begin(SERIAL_BAUDRATE);
    while (!Serial) {};
    Serial.println(F("********************************************"));
    Serial.println(F("Sleep-Modus konnte nicht aktiviert werden"));
    Serial.println(F("********************************************"));
    while (1 == 1) {
      delay (5000);
      Serial.print(".");
    };
  }

  digitalWrite(Power_Pin, HIGH);
  delay (5);
  rtc.begin();
  rtc.clearINTStatus();
  ok_sleep = true;

  Serial.begin(SERIAL_BAUDRATE);
  while (!Serial) {};
  Serial.println(F(" i.O."));
  Serial.println(" ");
  Serial.println(" ");
  Serial.flush();
  delay(5);
  if (power_off) {
    count++;
    WeckIntervallMinuten = WeckIntervallMinuten * 2;
  }

  // Power off test counter
  if (count == 2) {
    Serial.println(F(">>>>> Power - ON / OFF Test erfolgreich!  <<<<<"));
    Serial.println(" ");
    Serial.println(F("................. Test beendet!"));
    Serial.flush();
    Serial.end();        // Serial aus
    delay(50);
    digitalWrite(USB_RX, LOW);   // Port aus
    pinMode(USB_TX, INPUT);
    digitalWrite(USB_TX, LOW);   // Port aus
    digitalWrite(Power_Pin, LOW);

    TWCR &= ~(bit(TWEN) | bit(TWIE) | bit(TWEA));
    digitalWrite (A4, LOW);
    digitalWrite (A5, LOW);

    LowPower.powerDown(SLEEP_FOREVER, ADC_OFF, BOD_OFF);
    Serial.begin(SERIAL_BAUDRATE);
    while (!Serial) {};
    Serial.print(F(" It 'll never make it to this point! "));
    while (1 == 1) {
      delay (5000);
      Serial.print("*");
    }
  }
}
//--------------------------------------------------
