/*
   (C) 2025 R.Schick / Thorsten Gurzan - beelogger.de

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see <http://www.gnu.org/licenses/>.

*/

// beelogger.de - Arduino Datenlogger für Imker
// Erläuterungen dieses Programmcodes unter http://beelogger.de


//----------------------------------------------------------------
// Allgemeine Konfiguration
//----------------------------------------------------------------

#define UNI_2X  // Universal Platine Version 2.x // am Zeilenanfang entfernen


// Name des Zugangspunktes des Netzwerkproviders
const char APN[] PROGMEM = { "APN" };
static char APN_Benutzer[] = { "" };  // Benutzername des Zugangspunktes
static char APN_Passwort[] = { "" };  // Passwort des Zugangspunktes

//Beispiele:

//const char APN[] PROGMEM = {"internet.telekom"};
//static char APN_Benutzer[] = {"t-mobile"};
//static char APN_Passwort[] = {"tm"};


//const char APN[] PROGMEM = {"internet.t-mobile"};
//static char APN_Benutzer[] = {"t-mobile"};
//static char APN_Passwort[] = {"tm"};

//const char APN[] PROGMEM = {"web.vodafone.de"};
//const char APN[] PROGMEM = {"internet"};
//static char APN_Benutzer[] = {""};  // vodafone ohne APN-Benutzer/Passwort
//static char APN_Passwort[] = {""};

//const char APN[] PROGMEM = {"internet.eplus.de"};
//static char APN_Passwort[] = {"eplus"};
//static char APN_Benutzer[] = {"eplus"};


// Domainname zum Webserver mit beelogger-Skript
//const char serverName[] PROGMEM = {"meineDomain.de"};
const char serverName[] PROGMEM = { "community.beelogger.de" };

// Verzeichnis auf dem Webserver mit PHP-Skript für den jeweiligen beelogger
//const char beelogger_pfad[] PROGMEM = {"GET /Verzeichnis/beelogger1//beelogger_log.php?"};
const char beelogger_pfad[] PROGMEM = { "GET /USERX/beelogger1/beelogger_log.php?" };  // USERX  anpassen

// Passwort vom Webserver-Skript
//const char Passwort[] PROGMEM = {"meinPasswort"};
const char Passwort[] PROGMEM = { "Log" };  // your password;  "Log" = default testsystem beelogger

//----------------------------------------------------------------

// Option: Transfer Data via Port 443
#define _LTE_TLS 0  // 0 - Port 80, 1 - Port 443

// SIM76xx / A76xx Modul spezifische Parameter
#define A_7600 0          // 1 = A_7600E

//----------------------------------------------------------------
// Ende Konfiguration
//----------------------------------------------------------------



//----------------------------------------------------------------
// EE-Prom Adresse
uint8_t _EEProm_adress = 0x57;
//----------------------------------------------------------------

//----------------------------------------------------------------
// Konfiguration SIM7600 - Mobilfunk
//----------------------------------------------------------------
#include "SIM7600_beelogger.h"
#define SERVER_TM_OUT 10000  // 10000 = 10 sec
CGPRS_SIM7600 dfue;

// Pin  Belegung SIM 7600E
#define Stop_GSM_WLAN A2
//#define Stop_GSM_WLAN A3

#if (defined(ARDUINO_AVR_PRO) && (F_CPU == 16000000L) && (not defined UNI_2X))  // Pro Mini 5V 16MHz Universal 1.x
#undef Stop_GSM_WLAN
#define Stop_GSM_WLAN 12
#endif

uint8_t GSM_RX = 8;  // fix, not to modify
uint8_t GSM_TX = 9;  // fix, not to modify
//----------------------------------------------------------------

// stability test
#include <Wire.h>
#include "Sodaq_DS3231.h"
#define _DS3231_ADDRESS 0x68
#define Power_Pin 4

char ID_ID[] = "LTE_TEST_250803";

/****************************************************************/
// Variablen
/****************************************************************/
const float No_Val = 99.9f;    // Vorbelegung, Wert nicht gemessen
const float No_Value = -1.0f;  // Vorbelegung, Wert nicht gemessen

// Test-Messdaten (regulär kommen die Messdaten von den Sensoren)
float SensorTemp[2] = { 25.0, 35.0 };
float SensorFeuchte[2] = { 40.0, 60.0 };
float Licht = 5000;
float Gewicht[2] = { 2.5, No_Value };
float Batteriespannung = 4.7;
float Solarspannung = 1.7;
float Service = 0.0;
float Aux[3] = { 1.0, 2.0, 3.0 };
#define Aussenwerte 0
#define Beute1 1

bool OK = true;
uint8_t Verbindungsversuche = 0;

char DatenString[100];  // Datenbuffer
int call_len = 0;

//----------------------------------------------------------------
// Parameter Datenversand via Http // diese nicht modifizieren
//----------------------------------------------------------------
const char Str_Http[] PROGMEM = " HTTP/1.1\r\n";  // mit Leerzeichen vorne
const char Str_Content[] PROGMEM = "Content-Type: text/html\r\n";
const char Str_Con_alv[] PROGMEM = "Connection: close\r\n\r\n";
const char Str_Pw[] PROGMEM = { "PW=" };
const char Str_Opt[] PROGMEM = { "&Z=2" };
const char Str_Id[] PROGMEM = { "&ID=" };
const char Str_MData[] PROGMEM = { "&M1=" };  // Parameter Multi Data  Senden
#define SERVER_TM_OUT 10000                   // 10000 = 10 sec
//----------------------------------------------------------------


void setup() {
  char parameter[80];

  Serial.begin(9600);
  while (!Serial) {}
  Serial.println(ID_ID);
#if _LTE_TLS
  Serial.print(F("beelogger LTE SSL Mode Test "));
#endif
  Serial.println(F("\n beelogger LTE Test "));
#if defined(ARDUINO_AVR_PRO)
#if (F_CPU == 8000000L)  // 3.3V 8MHz
  Serial.println(F(" Pinbelegung: "));
  Serial.println(F("  LTE RX - PIN D8"));
  Serial.println(F("  LTE TX - PIN D9"));
  Serial.println("\n");
#endif
#endif
  digitalWrite(Power_Pin, HIGH);
  pinMode(Power_Pin, OUTPUT);


  Serial.print(F("Netzwerk ... "));
  Serial.print(parameter);
  Serial.print(" ... ");
  if (strcmp(parameter, "APN") == 0) {
    Serial.println(" ... ");
    Serial.println(F("Bitte Provider konfigurieren!"));
    Serial.flush();
    OK = false;
    LTE_Abschaltung();
  }

  strcpy_P(parameter, beelogger_pfad);
  char *p_buf = strstr(parameter, "/beelogger");  // search first /
  uint8_t n = atoi(p_buf + 10);
  //Serial.println(p_buf + 10);
  if (n == 0) {
    Serial.println(parameter);
    Serial.println(F("\nLTE Test Sketch nur mit beeloggerX,  X = 1...n"));
    Serial.println(F("keine DUO, Triple, Quad, usw."));
    Serial.flush();
    OK = false;
    LTE_Abschaltung();
  }

  Serial.println(F("\nTeste DS3231 vorhanden?"));
  Serial.flush();
  Wire.begin();
  Wire.beginTransmission(_DS3231_ADDRESS);
  uint8_t error = Wire.endTransmission();

  if (error == 0) {
    Serial.println("DS3231 ... gefunden\n");

    EE_WriteMem(0, ID_ID, strlen(ID_ID));
    EE_ReadStr(0, DatenString, strlen(ID_ID));
    //Serial.println(DatenString);
    if (strcmp(ID_ID, DatenString) == 0) Serial.println(F("EE-Prom ... i.O.\n"));
    else {
      Serial.println(F("###  EE-Prom Fehler  ###\n"));
      delay(10000);
    }
    rtc.begin();
    rtc.setDateTime(1);  // start RTC
  }


  if (OK) {  // pfad ok
    Serial.println(F("Initialisiere Modul ... "));
    Serial.flush();
    digitalWrite(Stop_GSM_WLAN, HIGH);
    pinMode(Stop_GSM_WLAN, OUTPUT);

    if (dfue.init()) {
      Serial.println("Init OK");
      Serial.println(" ");
      Serial.flush();
    } else {
      Serial.println(F("SIM7600E Modul konnte nicht initialisiert werden"));
      Serial.println(F("  SIM Karte ?"));
      Serial.println(F("  Konfiguration 9600Bd ?"));
      Serial.println(F("  Akkuspannung kontrollieren"));
      Serial.println(dfue.buffer);
      Serial.flush();

      if (dfue.sendCommand("AT") == 0) {
        Serial.println(F("  Fehler in SIM7600E RX / TX Verbindung ? "));
      } else {
        SIM7600_voltage();
      }

      Serial.println(" ");
      OK = false;
      LTE_Abschaltung();
    }
  }
  if (OK) {  // init ok
    dfue.sendCommand("AT+CGDCONT=1,\"IP\",\"\""); // reset any APN
    Serial.println(F("SIMCOM Modulinformationen:"));
    Serial.flush();
    dfue.sendCommand("AT+SIMCOMATI");
    Serial.println(dfue.buffer);
    Serial.flush();
    delay(1000);
    delay(1000);
    Serial.println(F("Verbindungsinformationen:"));
    Serial.flush();
    dfue.sendCommand("AT+CPSI?");
    Serial.println(dfue.buffer);
    Serial.println(" ");
    Serial.flush();
    delay(1000);
    Serial.println(F("Netzwerk/Provider Info:"));
    Serial.flush();
    dfue.sendCommand("AT+COPS?");
    Serial.println(dfue.buffer);
    Serial.println(" ");
    Serial.flush();
    delay(1000);
    Serial.println(" ");
    Serial.flush();
    delay(1000);
  }
  if (OK) {  // APN ok
    strcpy_P(parameter, APN);
    Serial.print(F("Verbinde zum Netzwerk ... "));
    Serial.print(parameter);
    Serial.print(" ... ");
    if (dfue.start(parameter, APN_Benutzer, APN_Passwort, A_7600) == 0) {
      Serial.println("Verbindung zum Netz OK");
      Serial.println(" ");
    } else {
      OK = false;
      Serial.println(F("Verbindung zum Netz konnte nicht aufgebaut werden"));
      Serial.println(" ");
      LTE_Abschaltung();
      while (1) {
        delay(100);
      };
    }
  }
  if (OK) {  // start ok
    delay(3000);
    char c = 'm';
    int Signal = dfue.getSignalQuality();
    if ((Signal > -10)) {
      Serial.print(F("Signalstärke unbrauchbar "));
      while (c != 'w') {
        c = Serial.read();
        delay(1000);
      }
    }
    if ((Signal <= -88)) {
      Serial.print(F("Signal schlecht : "));
      Serial.print(Signal);
      Serial.println(F(" dB"));
      while (c != 'w') {
        c = Serial.read();
        delay(1000);
      }
    } else {
      if (Signal) {
        Serial.print(F("Signalstaerke : "));
        Serial.print(Signal);
        Serial.println(F(" dB"));
      }
    }
  }
}

int count = 0;
//######################################
void loop() {

  if (OK) {
#if _LTE_TLS
    send_data_tls();
#else
    send_data_tcp();
#endif
  }  // if (OK)
}  // loop

void SIM7600_voltage() {
  if (dfue.sendCommand("AT+CBC", 5000)) {
    Serial.println("\n");
    Serial.print(F("SIM Versorgung : "));
    char *p = strstr(dfue.buffer, "+CBC: ");
    if (p) {
      Serial.print(p + 5);
    }
  }
}

void LTE_Abschaltung() {
  dfue.stop();
  Serial.println(" ");
  dfue.shutdown();
  digitalWrite(Stop_GSM_WLAN, LOW);
  digitalWrite(Power_Pin, LOW);
  Serial.println(F("LTE off"));
  Serial.println();
}

//####################################
void send_data_tcp() {
  char parameter[64];  // data  http call

  do {  // make GET calls
    strcpy_P(parameter, serverName);
    Serial.print(F("Verbinde zum Server: "));
    Serial.print(parameter);
    Serial.println(" ... ");
    Serial.flush();
    if (dfue.Connect(parameter) == 0) {  // Start TCP
      Serial.println(F("\n\rConnect Fehler"));
      OK = false;
    } else {
      Serial.println(F(" verbunden.\n"));

      if (dfue.prep_send()) {  // Open Server connection
        DatenString_erstellen();

        strcpy_P(parameter, beelogger_pfad);
        dfue.send(parameter);  //GET ...
        Serial.println(parameter);
        Serial.flush();

        strcpy_P(parameter, Str_Pw);
        dfue.send(parameter);
        strcpy_P(parameter, Passwort);
        dfue.send(parameter);
        strcpy_P(parameter, Str_Id);
        dfue.send(parameter);
        Serial.print(parameter);
        Serial.flush();

        strcpy(parameter, ID_ID);
        dfue.send(parameter);  // System ID
        Serial.print(parameter);
        Serial.flush();  //Daten
        strcpy_P(parameter, Str_Opt);
        dfue.send(parameter);
        Serial.println(parameter);
        Serial.flush();

        strcpy_P(parameter, Str_MData);
        dfue.send(parameter);
        Serial.print(parameter);
        Serial.flush();

        dfue.send(DatenString);  // Datensatz
        Serial.println(DatenString);
        Serial.flush();

        delay(50);

        strcpy_P(parameter, Str_Http);
        dfue.send(parameter);  // HTTP/1.1
        dfue.send("Host: ");
        strcpy_P(parameter, serverName);
        dfue.send(parameter);
        dfue.send("\r\n");
        strcpy_P(parameter, Str_Con_alv);
        dfue.send(parameter);
        dfue.send(0x00);  // Startkommando senden
        // http complete

        if (dfue.sendCommand(0, 10000, "ok *")) {     // hat Server Anfrage quittiert ?
          OK = true;                                  // success
          char *p_buf = strstr(dfue.buffer, "ok *");  // search string start
          byte cnt = 0;
          do {
            p_buf--;  // search start of line
            cnt++;
          } while ((!(*p_buf == '\n')) && (cnt < 20));

          Serial.print(F("Quittung : "));
          Serial.println(p_buf);
          Serial.println(F("Test erfolgreich\n"));
          Serial.flush();

          // Format Quittung aus beelogger-log.php
          // echo "5ok * "  = 5 Minuten
          Serial.println(F("OK"));
        } else {
          Serial.println(F("Fehler!"));
        }
        Serial.flush();
        dfue.disConnect();

        delay(200);
      }
      int Signal = dfue.getSignalQuality();
      if ((Signal > -10)) {
        Serial.println(F("Signalstärke unbrauchbar"));
      }
      if ((Signal <= -88)) {
        Serial.print(F("Signal schlecht : "));
        Serial.print(Signal);
        Serial.println(F(" dB"));
      } else {
        if (Signal) {
          Serial.print(F("Signalstaerke : "));
          Serial.print(Signal);
          Serial.println(F(" dB"));
        }
      }
      count++;
      if (count < 2) {
        // Test-Messdaten um 1 erhöhen
        SensorTemp[0] = 21.3;
        SensorTemp[1] += 0.3;
        SensorFeuchte[0]++;
        SensorFeuchte[1]++;
        Licht += 50.0;
        Gewicht[0] += 0.1;
        Batteriespannung += 0.2;
        Solarspannung += 0.1;
        Serial.println(" ");
        Serial.println(F(" wiederhole Senden in 4 Minuten"));
        Serial.flush();
        digitalWrite(Stop_GSM_WLAN, LOW);
        for (int i = 0; i < (6 * 4); i++) {
          delay(10000);
          Serial.print(".");
        }
        digitalWrite(Stop_GSM_WLAN, HIGH);
        delay(10000);
        Serial.print(F("Verbinde zum Netzwerk ... "));
        strcpy_P(parameter, APN);
        Serial.print(parameter);
        Serial.print(" ... ");
        if (dfue.start(parameter, APN_Benutzer, APN_Passwort) == 0) {
          Serial.println("Verbindung zum Netz OK");
          Serial.println(" ");
        } else {
          OK = false;
          Serial.println(F("Verbindung zum Netz konnte nicht aufgebaut werden"));
          Serial.println(" ");
          LTE_Abschaltung();
          while (1) {
            delay(100);
          };
        }
      }
    }  // prep_Send

  } while ((count < 2) && OK);
  // Abschaltung
  Serial.println(F("Test beendet"));
  OK = false;
  LTE_Abschaltung();
}

//####################################
void send_data_tls() {
  char parameter[64];  // data  http call

  if (dfue.ssl_init()) {  // Init SSL
    Serial.println(F("\n\rSSL Init!"));
    Serial.flush();

    if (dfue.ssl_start() == 0) {  // Start SSL
      Serial.println(F("\n\rSSL Start Fehler"));
      OK = false;
    } else {
      Serial.println(F("\n\rSSL Start!\n"));
      Serial.flush();


      strcpy_P(parameter, serverName);
      Serial.print(F("Verbinde zum Server: "));
      Serial.print(parameter);
      Serial.print(" ... ");
      Serial.flush();
      if (dfue.ssl_open(parameter) == 0) {  // Start SSL
        Serial.println(F("\n\rConnect Fehler"));
        OK = false;
      } else {
        Serial.println(F(" verbunden.\n"));

        DatenString_erstellen();

        strcpy_P(parameter, beelogger_pfad);  // GET ....
        dfue.ssl_data(parameter);
        Serial.println(parameter);
        Serial.flush();
        strcpy_P(parameter, Str_Pw);
        dfue.ssl_data(parameter);
        strcpy_P(parameter, Passwort);
        dfue.ssl_data(parameter);

        // System ID
        strcpy_P(parameter, Str_Id);
        dfue.ssl_data(parameter);
        Serial.print(parameter);
        Serial.flush();

        strcpy(parameter, ID_ID);
        dfue.ssl_data(parameter);
        Serial.print(parameter);
        Serial.flush();
        strcpy_P(parameter, Str_Opt);
        dfue.ssl_data(parameter);
        Serial.println(parameter);
        Serial.flush();

        //Daten
        strcpy_P(parameter, Str_MData);
        dfue.ssl_data(parameter);
        Serial.print(parameter);
        Serial.flush();

        dfue.ssl_data(DatenString);  // Datensatz
        delay(50);
        Serial.println(DatenString);
        Serial.flush();

        strcpy_P(parameter, Str_Http);  // HTTP/1.1
        dfue.ssl_data(parameter);
        dfue.ssl_data("Host: ");
        strcpy_P(parameter, serverName);
        dfue.ssl_data(parameter);
        dfue.ssl_data("\r\n");
        strcpy_P(parameter, Str_Content);
        dfue.ssl_data(parameter);
        strcpy_P(parameter, Str_Con_alv);
        dfue.ssl_data(parameter);
        // http complete

        if (dfue.sendCommand(0, 10000, "ok *")) {     // hat Server Anfrage quittiert ?
          OK = true;                                  // success
          char *p_buf = strstr(dfue.buffer, "ok *");  // search string start
          byte cnt = 0;
          do {
            p_buf--;  // search start of line
            cnt++;
          } while ((!(*p_buf == '\n')) && (cnt < 20));

          Serial.print(F("Quittung : "));
          Serial.println(p_buf);
          Serial.println(F("Test erfolgreich\n"));
          Serial.flush();

          // Format Quittung aus beelogger-log.php
          // echo "5ok * "  = 5 Minuten

          dfue.ssl_data("+++");       //dfue.sendCommand("+++", 2000); // back to command mode
          dfue.sendCommand(0, 3000);  // wait for connection terminate
          Serial.println(F("wait session close "));
          Serial.flush();
          if (!dfue.sendCommand(0, 3000, "PEER_CLOSED")) {  // wait for connection terminate
            dfue.ssl_close();
          }
          Serial.println(F("OK"));
        } else {
          OK = false;
          Serial.println("Fehler");
          dfue.ssl_close();
        }
        Serial.flush();
        // received ok *
      }
      delay(200);

      int Signal = dfue.getSignalQuality();
      if ((Signal > -10)) {
        Serial.println(F("Signalstärke unbrauchbar"));
      }
      if ((Signal <= -88)) {
        Serial.print(F("Signal schlecht : "));
        Serial.print(Signal);
        Serial.println(F(" dB"));
      } else {
        if (Signal) {
          Serial.print(F("Signalstaerke : "));
          Serial.print(Signal);
          Serial.println(F(" dB"));
        }
      }
      digitalWrite(Stop_GSM_WLAN, LOW);
      Serial.println(F("Test beendet"));
    }  // if SSL_Start
    OK = false;
    LTE_Abschaltung();
  }  // ssl init
  else {
    OK = false;
    Serial.println(F("Initialisierung fehlgeschlagen"));
    Serial.println(" ");
    Serial.println(dfue.buffer);
    Serial.println(" ");
    LTE_Abschaltung();
  }
}
/**
  @brief Funktion DatenString_erstellen()
  erstellt konfigurationsabhängigen Datensatz zum abspeichern im EE-Prom

  @param  none,        globale Variablen
  @return  none
*/
#include <Sodaq_DS3231.h>
void DatenString_erstellen() {

  int count = 0;
  DateTime jetzt = DateTime(__DATE__, __TIME__);
  count = sprintf(DatenString, "%d/", jetzt.year());
  count += sprintf(DatenString + count, "%2.2d/", jetzt.month());
  count += sprintf(DatenString + count, "%2.2d_", jetzt.date());
  count += sprintf(DatenString + count, "%2.2d:", jetzt.hour());
  count += sprintf(DatenString + count, "%2.2d:", jetzt.minute());
  count += sprintf(DatenString + count, "%2.2d", jetzt.second());

  count = Wert_hinzufuegen(count, SensorTemp[Beute1], 1, No_Val);            // Stocktemperatur 1
  count = Wert_hinzufuegen(count, SensorTemp[Aussenwerte], 1, No_Val);       // Außentemperatur
  count = Wert_hinzufuegen(count, SensorFeuchte[Beute1], 1, No_Value);       // Stockluftfeuchte 1
  count = Wert_hinzufuegen(count, SensorFeuchte[Aussenwerte], 1, No_Value);  // Außenluftfeuchte
  count = Wert_hinzufuegen(count, Licht, 1, No_Value);                       // Licht
  count = Wert_hinzufuegen(count, Gewicht[0], 2, No_Value);                  // Gewicht 1

  count = Wert_hinzufuegen(count, Batteriespannung, 2, No_Value);  // Akkuspannung
  count = Wert_hinzufuegen(count, Solarspannung, 2, No_Value);     // Solarspannung
  count = Wert_hinzufuegen(count, Service, 2, No_Value);           // Service
  count = Wert_hinzufuegen(count, Aux[1], 1, No_Value);            // Aux 1
  count = Wert_hinzufuegen(count, Aux[2], 2, No_Value);            // Aux 2
  count = Wert_hinzufuegen(count, Aux[0], 2, No_Value);            // Aux 0
  DatenString[count] = 0;
}


/**
  @brief Funktion Wert_hinzufuegen()
  erstellt Stringanteil für Datensatz

  @param
  count: aktuelle Position im String,
  Wert: Messwert,
  Nachkommastellen: Anzahl der Nachkommastellen für diesen Messwert
  Fehler: Vergleichswert für fehlerhaften oder unbenutzten Messwert
  @return  neue Position im Datenstring
*/
int Wert_hinzufuegen(int count, float Wert, uint8_t Nachkommastellen, float Fehler) {
  char Konvertierung[16];
  int count_neu = count;

  if (Wert == Fehler) {
    count_neu += sprintf(DatenString + count, ",%s", "");
  } else {
    dtostrf(Wert, 1, Nachkommastellen, Konvertierung);
    count_neu += sprintf(DatenString + count, ",%s", Konvertierung);
  }
  return count_neu;
}

// EE-Prom functions
uint8_t EE_WriteMem(int iAddr, const char *pBuf, int iCnt) {
  uint8_t iBytes, iRC = 0;
  // Writes are restricted to a single 32 byte page.  Therefore. if a write spans a page
  // boundry we must split the write.
  while (iCnt > 0) {
    iBytes = min(iCnt, BUFFER_LENGTH - 2);
    int iCurPage = iAddr & ~((int)0x1f);
    if (iAddr + iBytes > iCurPage + 32) {  // Number of bytes is too large
      iBytes = (iCurPage + 32) - iAddr;
    }
    Wire.beginTransmission(_EEProm_adress);
    Wire.write(highByte(iAddr));  // Address MSB
    Wire.write(lowByte(iAddr));   // Address LSB
    Wire.write((uint8_t *)pBuf, iBytes);
    Wire.endTransmission();
    iRC += (int)iBytes;
    iCnt -= (int)iBytes;
    iAddr += (int)iBytes;
    pBuf += (int)iBytes;
    delay(10);  // delay(50);  // Give the EEPROM time to write its data
  }             /* while */
  delay(20);
  return (iRC);
}

uint8_t EE_ReadStr(int iAddr, char Buf[], int iCnt) {
  uint8_t iRead = 0, iBytes;
  char c;
  while (iCnt > 0) {
    Wire.beginTransmission(_EEProm_adress);
    Wire.write(iAddr >> 8);    // Address MSB
    Wire.write(iAddr & 0xff);  // Address LSB
    Wire.endTransmission();
    iBytes = min(iCnt, 128);
    Wire.requestFrom(_EEProm_adress, iBytes);
    while (Wire.available() && iCnt > 0) {
      c = Wire.read();
      Buf[iRead] = c;
      if (c == '\0') {
        iCnt = 0;
        break;
      } /* if */
      iRead++;
      iCnt--;
      iAddr++;
    } /* while */
  }
  return (iRead);
}
