/*
   (C) 2024 R.Schick / Thorsten Gurzan - beelogger.de

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// beelogger.de - Arduino Datenlogger für Imker
// Erläuterungen dieses Programmcodes unter http://beelogger.de
// Version 2022    beelogger Multi mit EE-Prom, bis zu vier Waagen
// 30.04.2021 Status Reporting
// 03.05.2021 EE-Prom-only
// 14.11.2021 service switch handling
// 22.05.2022 swarm alarm
// 10.06.2022 swarm alarm intervall
// 10.12.2022 autoconfig Arduino ProMini
// 16.03.2023 preset Gewicht[]
// 01.04.2023 Universal Platine 2.x
// 06.04.2023 store dataset counter in eeprom
// 15.04.2023 check EE-Prom
// Version 2023
// 07.06.2023 Universal 2.x
// 30.06.2023 change startup batterie measurement
// 11.07.2023 Universal 2.x
// 19.04.2024 stop on startup error
// 30.06.2024 Zuordnung Aux[]
// 18.07.2024 Use EE-Prom 'round robin'
// 30.07.2024 check variable Size
// 30.11.2025 Watchdog I2C Daten
// 06.12.2025 Einschaltzeit Batterie low

char ID_ID[16] = "GSM_MR_260128";  //nur Grossbuchstaben,Zahlen, keine Blank

#define __ASSERT_USE_STDERR
#include <assert.h>
#include "Config_GSM.h"    // Konfiguration eintragen
#include "Config_MULTI.h"  // Konfiguration und Kalibrierung eintragen
#include "beelogger_config.h"
#include <avr/wdt.h>

/****************************************************************/
// Variablen
/****************************************************************/
//                   DS18B20_5, DHT1,   DHT2,   Si7021, SHT31a, SHT31b, BMEa,     BMEb, DS18B20_1,_2,_3,_4,
float SensorTemp[13] = { No_Val, No_Val, No_Val, No_Val, No_Val, No_Val, No_Val, No_Val, No_Val, No_Val, No_Val, No_Val, No_Val };

//                          NotUsed, DHT1,     DHT2,     Si7021,   SHT31a,   SHT31b,   BMEa,     BMEb,     DS18B20_a,_b,_c,_d,
float SensorFeuchte[12] = { No_Value, No_Value, No_Value, No_Value, No_Value, No_Value, No_Value, No_Value, No_Value, No_Value, No_Value, No_Value };

//              Luftdruck, ZählerRegen, DS3231Temp
float Aux[3] = { No_Value, No_Value, No_Value };  // needs a preset

float Licht = No_Value;
float DS_Temp;

float Gewicht[6] = { No_Value, No_Value, No_Value, No_Value, No_Value, No_Value };
float LetztesGewicht[6] = { No_Value, No_Value, No_Value, No_Value, No_Value, No_Value };

float Batteriespannung;
float Solarspannung;
float Service = 0.0;  // preset to indicate reset system

// Variablen permanente Daten
uint32_t time_on = 0;
uint8_t WeckIntervallMinuten = WeckIntervall_default;
uint8_t report_info = 1;     // 0 = time-On, 1 = Pegel
uint16_t data_start = 0;     // Anzahl Messungen
uint8_t my_counter = 0;      // Anzahl Messungen
uint8_t nr_to_measure = 1;   // Senden bei Start
uint8_t next_send_hour = 0;  //
uint8_t send_cycle = 1;      // Normaler Sendezyklus
uint8_t snd_dly = 0;         // Send delay
uint8_t stuck_switch = 0;    // Service Switch permanent on
uint8_t s_alarm = 0;         // Schwarmalarm
uint8_t akku_save = 0;

// Interrupt Variablen
volatile uint8_t ok_sleep = true;
char DatenString[_Daten_Satz_Len];  // Datenbuffer
/****************************************************************/


void setup() {

  Serial.begin(Serial_Baudrate);
  calldelay(300);
  Serial.println(ID_ID);
  Serial.flush();
  freeRam();

  assert(sizeof(ID_ID) < 20);
  assert(sizeof(serverName) < 32);
  assert(sizeof(beelogger_pfad) < 48);
  //----------------------------------------------------------------
  //  System on
  //----------------------------------------------------------------

  stop_DFUe_device();

  digitalWrite(Power_Pin, HIGH);  // Power On
  pinMode(Power_Pin, OUTPUT);
  calldelay(5);

  Spannungen_messen();
  if (Batteriespannung < VMinimum) {
    Serial.println(F(" - low"));
    Serial.flush();
    _stop_();
  }
  //----------------------------------------------------------------


  //----------------------------------------------------------------
  // Setup Gewicht
  //----------------------------------------------------------------
  Sensor_Gewicht(true);  // Startwert Gewicht holen
  //----------------------------------------------------------------


  //----------------------------------------------------------------
  //  Check EE-Prom
  //----------------------------------------------------------------
  Check_EE();
  //----------------------------------------------------------------


  //----------------------------------------------------------------
  //  Sleep Mode Interrupt
  //----------------------------------------------------------------
  pinMode(DS3231_Interrupt_Pin, INPUT_PULLUP);
  calldelay(5);
  //----------------------------------------------------------------

  //----------------------------------------------------------------
  //  Init Rain Sensor
  //----------------------------------------------------------------
  Init_Rain_Int();  // init Interrupt Regensensor
}


/****************************************************************/
void loop() {
  uint8_t alarm;
  uint8_t user_sw;

  System_On();
  debugbegin(Serial_Baudrate);
  calldelay(300);
  if (Spannungen_messen()) {
    debugprintlnF("\nLoop");
    display_time();
    debugflush();
    user_sw = User_Int();
    if (user_sw) {  // 0 = service switch not set or switch time is over
      Sensor_Temp_Zelle(0);
      Sensor_DHT();
      Sensor_DS18B20();

      wdt_enable(WDTO_8S);
      Sensor_Si7021();
      Sensor_SHT31();

      Sensor_BME280();
      //Sensor_BMP280();

      Sensor_Licht();
      wdt_disable();

      Sensor_Regen();  //Regenmesser
      Sensor_Temp_Zelle(1);
      alarm = Sensor_Gewicht(0);
      sw_alarm(alarm);
    } else {
      s_alarm = 0;  // reset swarmalarm when user switch
      Sensor_Gewicht(1);
    }
    Daten_Senden(user_sw);
    sw_alarm_int();
    print_info();
    freeRam();
  }
  Alarm_konfigurieren();

  debugend();
  Serial_rxtx_off();

  System_Off();
  SleepNow();
}

// handle diagnostic informations given by assertion and abort program execution:
void __assert(const char *__func, const char *__file, int __lineno, const char *__sexp) {
  // transmit diagnostic informations through serial link.
  // Serial.println(__func);
  Serial.println(__file);
  Serial.println(__lineno, DEC);
  Serial.println(__sexp);
  Serial.flush();
  // abort program execution.
  abort();
}
/**
  @brief ATmega328, 3,3V, 8MHz, i.e. Pro Mini
    check for ProMini and 8MHz and Debug-Flag
    give warning to user if myDEBUG is set
*/
#if defined(ARDUINO_AVR_PRO)
#if (F_CPU == 8000000L)
#if (myDEBUG == 1)
#warning in beelogger_config.h ist myDEBUG 1
#pragma message "myDEBUG 1 in beelogger_config.h nur für Testbetrieb"
#pragma message "myDEBUG 0 für Dauerbetrieb am Bienenstand"
#endif
#endif
#endif
