/*
 * (C) 2017 Thorsten Gurzan - beelogger.de
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// beelogger.de - Arduino Datenlogger für Imker
// Erläuterungen dieses Programmcodes unter http://beelogger.de
// Version 2021

//----------------------------------------------------------------
// Allgemeine Konfiguration
//----------------------------------------------------------------
uint32_t SpeicherIntervall = 300; // Speicherintervall / Versand an Master

byte Anzahl_Sensoren_DS18B20 = 0; // Mögliche Werte: '0','1','2'

byte Anzahl_Sensoren_DHT = 0; // Mögliche Werte: '0','1','2'

byte Anzahl_Sensoren_Licht = 0; // Mögliche Werte: '0','1'

byte Anzahl_Sensoren_Gewicht = 0; // Mögliche Werte: '0','1'

const byte Anzahl_Sensoren_Bienenzaehler = 0; // Mögliche Werte: '0','1','2','3','4'

byte DHT_Typ = 1; // Mögliche Werte: '1' = DHT21, '2' = DHT22

long Taragewicht = 1;  // Hier ist der Wert aus der Kalibrierung einzutragen
float Skalierung = 1;  // Hier ist der Wert aus der Kalibrierung einzutragen

float Kalibriertemperatur = 0;       // Temperatur zum Zeitpunkt der Kalibrierung
float KorrekturwertGrammproGrad = 0; // Korrekturwert zur Temperaturkompensation - '0' für Deaktivierung
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration RF433 - Funk
//----------------------------------------------------------------
#include <VirtualWire.h>
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration DS18B20 - Temperatur
//----------------------------------------------------------------
#include <OneWire.h> 
#include <DallasTemperature.h> 
 
#define ONE_WIRE_BUS 3
#define Sensor_Aufloesung 12
DeviceAddress Sensor_Adressen;
 
OneWire oneWire(ONE_WIRE_BUS); 
DallasTemperature sensors(&oneWire); 
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration DHT21 / DHT22 - Temperatur und Luftfeuchte
//----------------------------------------------------------------
#include <dht.h> 
dht DHT;

byte DHT_Sensor_Pin[2] = {5,6};
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration Beleuchtungsstärke
//----------------------------------------------------------------
#include <Wire.h>
#include <AS_BH1750.h>

AS_BH1750 sensor;
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration Gewicht
//----------------------------------------------------------------
#include <HX711.h>              
#define HX711_SCK  A0
#define HX711_DT   A1     
HX711 scale; 
//----------------------------------------------------------------


//----------------------------------------------------------------
//Konfiguration Bienenzähler
//----------------------------------------------------------------
#include <Adafruit_MCP23017.h>
Adafruit_MCP23017 mcp[Anzahl_Sensoren_Bienenzaehler];
//----------------------------------------------------------------


//----------------------------------------------------------------
// Variablen
//----------------------------------------------------------------
float TempIn = 999.99;
float TempOut = 999.99;
float FeuchteIn = 999.99;
float FeuchteOut = 999.99;
float Licht = -1;
long Gewicht = 999999;
long LetztesGewicht = 0;
long BienenIn = -1;
long BienenOut = -1;

byte SensorAktuell[16];
byte LetzterSensor[Anzahl_Sensoren_Bienenzaehler][16]; 

float Temperatur[2] = {999.99,999.99};
float Luftfeuchte[2] = {999.99,999.99};

uint32_t LetztesIntervall = 0;
uint32_t LogDelayZeit;

int ID;
char msg[80];
//----------------------------------------------------------------



void setup() {

//----------------------------------------------------------------
// Setup RF433 - Funk
//----------------------------------------------------------------    
vw_set_tx_pin(2); 
vw_setup(2000);	  
//----------------------------------------------------------------  


//----------------------------------------------------------------
// Setup DS18B20 - Temperatur
//----------------------------------------------------------------  
if ((Anzahl_Sensoren_DS18B20 > 0) and (Anzahl_Sensoren_DS18B20 < 3)) {
  sensors.begin();
 
  for(byte i=0 ;i < sensors.getDeviceCount(); i++) {
    if(sensors.getAddress(Sensor_Adressen, i)) {
      sensors.setResolution(Sensor_Adressen, Sensor_Aufloesung);
    }
  }
}
//----------------------------------------------------------------
 
 
//----------------------------------------------------------------
// Setup DHT21 / DHT22 - Temperatur und Luftfeuchte
//----------------------------------------------------------------

//----------------------------------------------------------------
 
 
//----------------------------------------------------------------
// Setup Beleuchtungsstärke
//----------------------------------------------------------------
if (Anzahl_Sensoren_Licht == 1) {
  sensor.begin();
}
//---------------------------------------------------------------- 


//----------------------------------------------------------------
// Setup Gewicht
//----------------------------------------------------------------
if (Anzahl_Sensoren_Gewicht == 1) {
  scale.begin(HX711_DT, HX711_SCK,128);
  scale.set_offset(Taragewicht);  
  scale.set_scale(Skalierung);
  scale.power_up();
  scale.read();
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Setup Bienenzähler
//----------------------------------------------------------------
if ((Anzahl_Sensoren_Bienenzaehler > 0) and (Anzahl_Sensoren_Bienenzaehler < 5)) {
  for (byte j=0; j < Anzahl_Sensoren_Bienenzaehler; j++) {
    mcp[j].begin(j+4); 
    for (byte i=0; i < 16; i++) {
      mcp[j].pinMode(i, INPUT);
      mcp[j].pullUp(i, LOW);
      LetzterSensor[j][i] = 0;
    }   
  }  
}
//----------------------------------------------------------------
  
}


void loop() {
  Sensor_Bienen();
  if ((millis() - LetztesIntervall) >= (SpeicherIntervall*1000)) {
    Sensor_DS18B20();
    Sensor_Bienen();
    Sensor_DHT();
    Sensor_Bienen();
    Sensor_Licht();
    Sensor_Bienen();
    Sensor_Gewicht();
    Sensor_Bienen();
    Senden_RF433();  
    LetztesIntervall = millis(); 
  }
}


//----------------------------------------------------------------
// Funktion RF433 - Funk
//----------------------------------------------------------------
void Senden_RF433() {
  ID = random(1000,9999);
  
  for (int i=0; i < 5; i++) {  // Hier kann die Anzahl der zusätzlichen Aussendungen pro Datensatz eingestellt werden
    sprintf(msg, "%d,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld", ID, (long)TempIn*100, (long)TempOut*100, (long)FeuchteIn*100, (long)FeuchteOut*100, (long)Licht*100, (long)Gewicht*100, BienenIn, BienenOut);
    delay(5); 
    vw_send((uint8_t *)msg, strlen(msg));
    vw_wait_tx();
    delay(100);  
  }  
  if (Anzahl_Sensoren_Bienenzaehler != 0) {
    BienenIn = 0;
    BienenOut = 0;
  } 
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion DS18B20 - Temperatur
//----------------------------------------------------------------
void Sensor_DS18B20() {
  if ((Anzahl_Sensoren_DS18B20 > 0) and (Anzahl_Sensoren_DS18B20 < 3)) {
    sensors.requestTemperatures();
  
    for(byte i=0 ;i < Anzahl_Sensoren_DS18B20; i++) {
      if (i < sensors.getDeviceCount()) {
        for(byte j=0 ;j < 3; j++) { 
          if (j > 0 ) Sensor_Bienen(); 
          Temperatur[i] = sensors.getTempCByIndex(i);
          if ((Temperatur[i] < 60) and (Temperatur[i] > -40)) j=10; // Werte für Fehlererkennung
          else {
            Temperatur[i] = 999.99;
            LogDelay(1000);
          }
        } 
      }
    }  
    TempIn = Temperatur[0];                                     // Hier kann die Zuordnung der Sensoren geändert werden
    if (Anzahl_Sensoren_DS18B20 == 2) TempOut = Temperatur[1];  // Hier kann die Zuordnung der Sensoren geändert werden
  }
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion DHT21 / DHT22 - Temperatur und Luftfeuchte
//----------------------------------------------------------------
void Sensor_DHT() {
  if ((Anzahl_Sensoren_DHT > 0) and (Anzahl_Sensoren_DHT < 3)) {
    for(byte i=0 ;i < Anzahl_Sensoren_DHT; i++) {
      int check;
      
      for(byte j=0 ;j < 3; j++) {
        if (j > 0) LogDelay(2000);
        
        if (DHT_Typ == 1) check = DHT.read21(DHT_Sensor_Pin[i]); 
        else check = DHT.read22(DHT_Sensor_Pin[i]); 
        
        Temperatur[i] = 999.99;
        Luftfeuchte[i] = 999.99;
        
        switch (check) {
          case DHTLIB_OK:
            Luftfeuchte[i] = DHT.humidity; 
            Temperatur[i] = DHT.temperature;
            j=10;
          break;
        }  
      }
      TempOut = Temperatur[0];         // Hier kann die Zuordnung der Sensoren geändert werden
      FeuchteOut = Luftfeuchte[0];     // Hier kann die Zuordnung der Sensoren geändert werden
      if (Anzahl_Sensoren_DHT == 2) {
        TempIn = Temperatur[1];        // Hier kann die Zuordnung der Sensoren geändert werden
        FeuchteIn = Luftfeuchte[1];    // Hier kann die Zuordnung der Sensoren geändert werden
      }
    }
  }
}  
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion Beleuchtungsstärke
//----------------------------------------------------------------
void Sensor_Licht() {
  if (Anzahl_Sensoren_Licht == 1) {
    for(byte j=0 ;j < 3; j++) {  
      Licht = sensor.readLightLevel();
      if (Licht >= 0) j=10;
      else LogDelay(1000);
    }
  }  
}  
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion Gewicht
//----------------------------------------------------------------
void Sensor_Gewicht() {
  if (Anzahl_Sensoren_Gewicht == 1) {
    delay(100);
    for(byte j=0 ;j < 3; j++) { // Anzahl der Widerholungen, wenn Abweichung zum letzten Gewicht zu hoch
      Gewicht= scale.get_units(10);
      if ((Gewicht-LetztesGewicht < 500) and (Gewicht-LetztesGewicht > -500)) j=10; // Abweichung für Fehlererkennung
      if (j < 3) {
        LogDelay(3000);  // Wartezeit zwischen Wiederholungen      
      }
    } 
    // Temperaturkompensation
    if ((TempOut != 999.99)){
      if (TempOut > Kalibriertemperatur) Gewicht = Gewicht-(fabs(TempOut-Kalibriertemperatur)*KorrekturwertGrammproGrad); 
      if (TempOut < Kalibriertemperatur) Gewicht = Gewicht+(fabs(TempOut-Kalibriertemperatur)*KorrekturwertGrammproGrad);
    } 
    
    LetztesGewicht = Gewicht;
  }
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion Bienenzähler
//----------------------------------------------------------------
void Sensor_Bienen() {
  if ((Anzahl_Sensoren_Bienenzaehler > 0) and (Anzahl_Sensoren_Bienenzaehler < 5)) {  
    for (byte j = 0; j < Anzahl_Sensoren_Bienenzaehler; j++) {     
      
      for (byte i = 0; i < 16; i++) { 
        SensorAktuell[i] = mcp[j].digitalRead(i);
      } 
    
      for (byte i = 0; i < 8; i++) { 
        if (SensorAktuell[i] != LetzterSensor[j][i]) {
          if (SensorAktuell[i] > SensorAktuell[i+8]) {
            if (BienenIn == -1) BienenIn = 1;
            else  BienenIn++;
          }
        }
        if (SensorAktuell[i+8] != LetzterSensor[j][i+8]){ 
          if (SensorAktuell[i+8] > SensorAktuell[i]) {
            if (BienenOut == -1) BienenOut = 1;
            else BienenOut++;
          }
        }          
        LetzterSensor[j][i] = SensorAktuell[i];
        LetzterSensor[j][i+8] = SensorAktuell[i+8];
      }
    }  
    delay(10);
  }  
}    
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion LogDelay
//----------------------------------------------------------------
void LogDelay(uint32_t Zeit) {
  LogDelayZeit = millis(); 
  
  while ((millis() - LogDelayZeit) < (Zeit)) {
    Sensor_Bienen();
  } 
}  
//----------------------------------------------------------------
