/*
   (C) 2017 Thorsten Gurzan - beelogger.de

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// beelogger.de - Arduino Datenlogger für Imker
// Erläuterungen dieses Programmcodes unter http://beelogger.de
// Version 2021

#include <SPI.h>

//----------------------------------------------------------------
// Allgemeine Konfiguration
//----------------------------------------------------------------
#define myDEBUG 1


#define WeckIntervallMinuten  10 // Standard Weckintervall
#define Intervall_Winter      0 // 0 oder 1, wenn 1 dann Nachts und von August bis März Intervall im Stundentakt
#define AlternativIntervallMinuten  180
const float VMinimum = 3.6;          // Minimale Spannung ab der ab der keine Messungen und auch kein Versand von Daten erfolgt
const float VAlternativ = 3.65;      // Minimale Spannung ab der automatisch das alternative Intervall aktiviert wird

// wenn kein Sensor für TempOut gesetzt ist, wird automatisch der Temperatursensor der RTC verwendet
#define Anzahl_Sensoren_DS18B20  1 // Mögliche Werte: '0','1','2'

#define Anzahl_Sensoren_DHT  1    // Mögliche Werte: '0','1','2'

#define Anzahl_Sensoren_Licht  1  // Mögliche Werte: '0','1'

#define Anzahl_Sensoren_Gewicht 0 // Mögliche Werte: '0','1'

const long Taragewicht = 1;  // Hier ist der Wert aus der Kalibrierung einzutragen
const float Skalierung = 1.0; // Hier ist der Wert aus der Kalibrierung einzutragen

const float Kalibriertemperatur      = 0.0f;    // Temperatur zum Zeitpunkt der Kalibrierung
const float KorrekturwertGrammproGrad = 0.0f;   // Korrekturwert zur Temperaturkompensation - '0' für Deaktivierung

const long Kalib_Spannung = 100;  // Hier muss der Wert aus der Kalibrierung eingetragen werden, sonst funktioniert der Programmcode nicht
const long Kalib_Bitwert = 1000;   // Hier muss der Wert aus der Kalibrierung eingetragen werden, sonst funktioniert der Programmcode nicht
//----------------------------------------------------------------

//----------------------------------------------------------------
// Konfiguration Monitor via Serial / FTDI / USB
//----------------------------------------------------------------
#define Serial_Baudrate 9600

// Pins der serielle Schnittstelle
#define USB_RX      0          // Pin 0 RX-USB
#define USB_TX      1          // Pin 1 TX-USB

//----------------------------------------------------------------
// Konfiguration nRF24L01 - Funk
//----------------------------------------------------------------
#include <RF24.h>

#define ClientNummer  1  // Mögliche Werte: 1-6

#define nRF24_CSN  7    // Pin Belegung nRF24
#define nRF24_CE   8
#define nRF24_MOSI 11
#define nRF24_MISO 12
#define nRF24_SCK  13

static const uint64_t pipes[6] = {0xF0F0F0F0E1LL, 0xF0F0F0F0D2LL, 0xF0F0F0F0C3LL, 0xF0F0F0F0B4LL, 0xF0F0F0F0A5LL, 0xF0F0F0F096LL};
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration DS18B20 - Temperatur
//----------------------------------------------------------------
#include <OneWire.h>
#include <DallasTemperature.h>

#define ONE_WIRE_BUS 3
#define Sensor_Aufloesung 12
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration DHT21 / DHT22 - Temperatur und Luftfeuchte
//----------------------------------------------------------------
#include <dht.h>

byte DHT_Sensor_Pin[2] = {5, 6};
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration Beleuchtungsstärke
//----------------------------------------------------------------
#include <Wire.h>
#include <AS_BH1750.h>
//----------------------------------------------------------------


//----------------------------------------------------------------
// Konfiguration Gewicht
//----------------------------------------------------------------
#include <HX711.h>
#define HX711_SCK     A0    // HX711 S-Clock
#define HX711_DT      A1    // HX711 Data
//----------------------------------------------------------------


//----------------------------------------------------------------
//Konfiguration RTC & Sleep-Mode
//----------------------------------------------------------------
#include <Sodaq_DS3231.h>
#include <LowPower.h>

#define DS3231_Interrupt_Pin 2
#define Power_Pin 4
//----------------------------------------------------------------


//----------------------------------------------------------------
// Variablen
//----------------------------------------------------------------
#define No_Val  999.99f  // Vorbelegung, Wert nicht gemessen 
float TempIn = No_Val;
float TempOut = No_Val;
float FeuchteIn = No_Val;
float FeuchteOut = No_Val;
float Licht = -1;
long Gewicht = 999999;
long LetztesGewicht = 0;

float Batteriespannung = No_Val;
float Solarspannung = No_Val;

volatile bool ok_sleep = false;
//----------------------------------------------------------------



void setup() {

#if myDEBUG
  Serial.begin(Serial_Baudrate);
  Serial.println(F("Solar_nRF24L01_080419"));
#endif

  //----------------------------------------------------------------
  //  Setup RTC & Sleep-Mode
  //----------------------------------------------------------------
  digitalWrite(Power_Pin, HIGH);
  pinMode(Power_Pin, OUTPUT);

  delay(5);

  rtc.begin();

  DateTime pc_tim = DateTime(__DATE__, __TIME__);
  long l_pczeit = pc_tim.get();
  DateTime aktuell = rtc.now();
  long l_zeit = aktuell.get();
  if (l_pczeit > l_zeit)  rtc.setDateTime(l_pczeit);

  rtc.clearINTStatus();

  pinMode(DS3231_Interrupt_Pin, INPUT_PULLUP);
  ok_sleep = true;

  delay(5);
#if myDEBUG
  Serial.print(F("Systemzeit: "));
  Serial.print(aktuell.hour());
  Serial.print((":"));
  Serial.print(aktuell.minute());
  Serial.print((":"));
  Serial.println(aktuell.second());
  Serial.flush();
#endif
  //----------------------------------------------------------------


  //----------------------------------------------------------------
  // Setup nRF24L01 - Funk
  //----------------------------------------------------------------

  //----------------------------------------------------------------


  //----------------------------------------------------------------
  // Setup DS18B20 - Temperatur
  //----------------------------------------------------------------

  //----------------------------------------------------------------


  //----------------------------------------------------------------
  // Setup DHT21 / DHT22 - Temperatur und Luftfeuchte
  //----------------------------------------------------------------

  //----------------------------------------------------------------


  //----------------------------------------------------------------
  // Setup Beleuchtungsstärke
  //----------------------------------------------------------------

  //----------------------------------------------------------------


  //----------------------------------------------------------------
  // Setup Gewicht
  //----------------------------------------------------------------
  Sensor_Gewicht(true);
  //----------------------------------------------------------------

}


void loop() {
#if myDEBUG
  Serial.begin(Serial_Baudrate);
  Serial.println(F(" Loop"));
  Serial.flush();
#endif
  Spannungen_messen();
  if (Batteriespannung > VMinimum) {
    TempOut = No_Val;
    Sensor_DS18B20();
    Sensor_DHT();
    if ((TempOut == No_Val)) TempOut = rtc.getTemperature();
    Sensor_Licht();
    Sensor_Gewicht(false);
    Senden_nRF24L01();
  }
  Alarm_konfigurieren();
#if myDEBUG
  Serial.flush();
  Serial.end();        // Serial aus
  delay(50);
  digitalWrite(USB_RX, LOW);   // Port aus
  pinMode(USB_TX, INPUT);
  digitalWrite(USB_TX, LOW);   // Port aus
#endif
  SleepNow();
}


//----------------------------------------------------------------
// Funktion nRF24L01 - Funk
//----------------------------------------------------------------
void Senden_nRF24L01() {
  RF24 radio(nRF24_CE, nRF24_CSN);
#if myDEBUG
  Serial.println(F(" nRF send "));
  Serial.flush();
#endif
  radio.begin();
  delay(20);
  radio.setChannel(50);                // Funkkanal - Mögliche Werte: 0 - 127
  radio.setAutoAck(0);
  radio.setRetries(15, 15);
  radio.setPALevel(RF24_PA_HIGH);     // Sendestärke darf die gesetzlichen Vorgaben des jeweiligen Landes nicht überschreiten!
  // RF24_PA_MIN=-18dBm, RF24_PA_LOW=-12dBm, RF24_PA_MED=-6dBM, and RF24_PA_HIGH=0dBm

  radio.openWritingPipe(pipes[ClientNummer - 1]);
  radio.openReadingPipe(1, pipes[0]);

  radio.startListening();
  delay(20);

  long message[8] = {long(TempIn * 100), long(TempOut * 100), long(FeuchteIn * 100), long(FeuchteOut * 100), long(Licht * 100), Gewicht, long(Batteriespannung * 100), long(Solarspannung * 100)};
  radio.stopListening();
  radio.write(&message, sizeof(message));
  radio.powerDown();
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion DS18B20 - Temperatur
//----------------------------------------------------------------
void Sensor_DS18B20() {
#if myDEBUG
  Serial.print(F(" DS18B20: "));
  Serial.println(Anzahl_Sensoren_DS18B20);
  Serial.flush();
#endif
  if ((Anzahl_Sensoren_DS18B20 > 0) and (Anzahl_Sensoren_DS18B20 < 3)) {
    float Temperatur_DS;
    DeviceAddress Sensor_Adressen;
    OneWire oneWire(ONE_WIRE_BUS);
    DallasTemperature sensors(&oneWire);

    sensors.begin();

    for (byte i = 0 ; i < sensors.getDeviceCount(); i++) {
      if (sensors.getAddress(Sensor_Adressen, i)) {
        sensors.setResolution(Sensor_Adressen, Sensor_Aufloesung);
      }
    }

    sensors.requestTemperatures();

    for (byte i = 0 ; i < Anzahl_Sensoren_DS18B20; i++) {
      if (i < sensors.getDeviceCount()) {
        Temperatur_DS = sensors.getTempCByIndex(i);
        if (Temperatur_DS == DEVICE_DISCONNECTED_C) {
          Temperatur_DS = No_Val;
        }
      }
      // der erste DS18B20
      if (i == 0) {
        TempIn = Temperatur_DS;  // Hier kann die Zuordnung der Sensoren geändert werden
        //TempOut = Temperatur_DS;  // Hier kann die Zuordnung der Sensoren geändert werden
      }
      // der zweite DS18B20
      if (i == 1) {
        //TempIn = Temperatur_DS;  // Hier kann die Zuordnung der Sensoren geändert werden
        //TempOut = Temperatur_DS;  // Hier kann die Zuordnung der Sensoren geändert werden
      }
#if myDEBUG
      Serial.print(F(" Temperatur "));
      Serial.print(i + 1);
      Serial.print(F(" [C]: "));
      Serial.println(Temperatur_DS);
#endif
    }
  }
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion DHT21 / DHT22 - Temperatur und Luftfeuchte
//----------------------------------------------------------------
void Sensor_DHT() {
#if myDEBUG
  Serial.print(F(" DHT:  "));
  Serial.println(Anzahl_Sensoren_DHT);
  Serial.flush();
#endif
  if ((Anzahl_Sensoren_DHT > 0) and (Anzahl_Sensoren_DHT < 3)) {
    float Temperatur_DHT;
    float Luftfeuchte_DHT;
    int check;
    dht beeDHT;

    for (byte i = 0 ; i < Anzahl_Sensoren_DHT; i++) {

      check = beeDHT.read(DHT_Sensor_Pin[i]);

      if (check == DHTLIB_OK) {
        Luftfeuchte_DHT = beeDHT.humidity;
        Temperatur_DHT = beeDHT.temperature;
      }
      else {
        Temperatur_DHT = No_Val;
        Luftfeuchte_DHT = No_Val;
      }
      if (i == 0) { // der erste DHT
        //TempIn = Temperatur_DHT;        // Hier kann die Zuordnung der Sensoren geändert werden
        FeuchteIn = Luftfeuchte_DHT;    // Hier kann die Zuordnung der Sensoren geändert werden
        TempOut = Temperatur_DHT;       // Hier kann die Zuordnung der Sensoren geändert werden
        //FeuchteOut = Luftfeuchte_DHT;   // Hier kann die Zuordnung der Sensoren geändert werden
      }
      if (i == 1) { // der zweite DHT
        //TempIn = Temperatur_DHT;        // Hier kann die Zuordnung der Sensoren geändert werden
        //FeuchteIn = Luftfeuchte_DHT;    // Hier kann die Zuordnung der Sensoren geändert werden
        //TempOut = Temperatur_DHT;       // Hier kann die Zuordnung der Sensoren geändert werden
        //FeuchteOut = Luftfeuchte_DHT;   // Hier kann die Zuordnung der Sensoren geändert werden
      }
#if myDEBUG
      Serial.print(F(" Temperatur [C]: "));
      Serial.println(Temperatur_DHT);
      Serial.print(F(" Feuchte [%RH]: "));
      Serial.println(Luftfeuchte_DHT);
#endif
    }
  }
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion Beleuchtungsstärke
//----------------------------------------------------------------
void Sensor_Licht() {
#if myDEBUG
  Serial.print(F(" BH1750: "));
  Serial.println(Anzahl_Sensoren_Licht);
  Serial.flush();
#endif
  if (Anzahl_Sensoren_Licht == 1) {
    AS_BH1750 sensor;
    if (sensor.begin()) {
      Licht = sensor.readLightLevel();
      sensor.powerDown();
    }
#if myDEBUG
    Serial.print(F(" Licht [lux]: "));
    Serial.println(Licht);
#endif
  }
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion Gewicht
//----------------------------------------------------------------
void Sensor_Gewicht(boolean quick) {
#if myDEBUG
  Serial.print(F(" Waage Anzahl: "));
  Serial.println(Anzahl_Sensoren_Gewicht);
  Serial.flush();
#endif
  if (Anzahl_Sensoren_Gewicht == 1) {
    const long Diff_Gewicht = 500;

    HX711 scale;
    scale.begin(HX711_DT, HX711_SCK);
    scale.set_offset(Taragewicht);
    scale.set_scale(Skalierung);
    scale.power_up();
    LowPower.powerStandby(SLEEP_1S, ADC_OFF, BOD_OFF); // Wartezeit
    scale.read();
    for (byte j = 0 ; j < 3; j++) { // Anzahl der Wiederholungen, wenn Abweichung zum letzten Gewicht zu hoch
      Gewicht = (long) scale.get_units(10);
      if (quick) {
        LetztesGewicht = Gewicht;
        break;
      }
      if (abs(Gewicht - LetztesGewicht) < Diff_Gewicht) break; // Abweichung für Fehlererkennung
      LowPower.powerStandby(SLEEP_2S, ADC_OFF, BOD_OFF); // Wartezeit zwischen Wiederholungen
      LowPower.powerStandby(SLEEP_1S, ADC_OFF, BOD_OFF); // Wartezeit zwischen Wiederholungen
    }
    // Temperaturkompensation
    if (TempOut != No_Val) {
      Gewicht = Gewicht - (long)((TempOut - Kalibriertemperatur) * KorrekturwertGrammproGrad);
    }
    LetztesGewicht = Gewicht;
    scale.power_down();
#if myDEBUG
    Serial.print(F(" Gewicht [kg]: "));
    Serial.println(Gewicht);
    Serial.flush();
#endif
  }
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion Spannungen messen
//----------------------------------------------------------------
void Spannungen_messen() {
  int Messung_Spannung = analogRead(A6);
  Messung_Spannung = 0;
  for (byte j = 0 ; j < 16; j++) {
    Messung_Spannung += analogRead(A6);
  }
  Messung_Spannung = Messung_Spannung >> 2;
  Batteriespannung = (map(Messung_Spannung, 0, Kalib_Bitwert, 0, Kalib_Spannung)) / 1000.0;
#if myDEBUG
  Serial.print(F(" Batterie [V]: "));
  Serial.println(Batteriespannung);
  Serial.flush();
#endif
  if (Batteriespannung > VMinimum) {
    Messung_Spannung = analogRead(A7);
    Messung_Spannung = 0;
    for (byte j = 0 ; j < 16; j++) {
      Messung_Spannung += analogRead(A7);
    }
    Messung_Spannung = Messung_Spannung >> 2;
    Solarspannung = (map(Messung_Spannung, 0, Kalib_Bitwert, 0, Kalib_Spannung)) / 1000.0;

  }
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion Alarm konfigurieren
//----------------------------------------------------------------
void Alarm_konfigurieren() {
  int IntervallMinuten;
  DateTime aktuell = rtc.now();
  boolean y = false;         // Arduino bis zur vollen Stunde schlafen lassen.

  if (Batteriespannung > VAlternativ) {
    IntervallMinuten = WeckIntervallMinuten;
    if (Intervall_Winter) {
      if ((aktuell.month() > 3) && (aktuell.month() < 8)) {       // April, Mai, Juni, Juli
        if ( (aktuell.hour() > 6) && (aktuell.hour() < 18) ) {    // zwischen 7 und 18 Uhr
          y = true;      // Abfrage alleIntervallminuten
        }
      }
      if (y == false) {  // ansonsten im Stundenzyklus
        rtc.enableInterrupts(EveryHour);  //interrupt at full hour
        return;
      }
    }
  } else {
    IntervallMinuten = AlternativIntervallMinuten;
  }
  long l_tm = aktuell.get();
  aktuell = l_tm + IntervallMinuten * 60; // xx Minuten später
#if myDEBUG
  Serial.print(F("Wakeup at: "));
  Serial.print(aktuell.hour());
  Serial.print((":"));
  Serial.print(aktuell.minute());
  Serial.print((":"));
  Serial.println(aktuell.second());
  Serial.flush();
#endif
  l_tm = aktuell.minute() - aktuell.minute() % IntervallMinuten;  // Auf Minuten runden
  rtc.enableInterrupts(aktuell.hour(), l_tm, 0); //interrupt at  hour, minute, second
}
//----------------------------------------------------------------



//----------------------------------------------------------------
// Funktion SleepNow
//----------------------------------------------------------------
void SleepNow() {

  digitalWrite(Power_Pin, LOW);
  delay(5);

  TWCR &= ~(bit(TWEN) | bit(TWIE) | bit(TWEA));
  digitalWrite (A4, LOW);
  digitalWrite (A5, LOW);
  digitalWrite (nRF24_CSN, LOW);
  digitalWrite (nRF24_CE, LOW);
  digitalWrite (ONE_WIRE_BUS, LOW);
  digitalWrite (DHT_Sensor_Pin[0], LOW);
  digitalWrite (DHT_Sensor_Pin[1], LOW);
  digitalWrite (HX711_SCK, LOW);
  pinMode (nRF24_MOSI, INPUT);

  attachInterrupt(0, WakeUp, LOW);

  delay(1);
  if (ok_sleep) {
    LowPower.powerDown(SLEEP_FOREVER, ADC_OFF, BOD_OFF);
  }

  pinMode (nRF24_MOSI, OUTPUT);
  digitalWrite (nRF24_CSN, HIGH);
  digitalWrite (nRF24_CE, HIGH);
  digitalWrite (HX711_SCK, HIGH);
  digitalWrite (DHT_Sensor_Pin[0], HIGH);
  digitalWrite (DHT_Sensor_Pin[1], HIGH);

  digitalWrite(Power_Pin, HIGH);
  delay (5);

#if myDEBUG
  Serial.begin(Serial_Baudrate);
  Serial.println(F(" Sleep ende."));
  Serial.flush();
#endif
  rtc.begin();
  rtc.clearINTStatus();
  while (digitalRead(DS3231_Interrupt_Pin) == false) {};
  ok_sleep = true;

  delay(50);
}
//----------------------------------------------------------------


//----------------------------------------------------------------
// Funktion WakeUp
//----------------------------------------------------------------
void WakeUp() {
  ok_sleep = false;
  detachInterrupt(0);
}
//----------------------------------------------------------------
